<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use App\Enums\UserRole;
use App\Enums\DeletedType;
use App\Enums\PostAvailableType;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Facades\Hash;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'deleted_type',
        'last_access',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'role' => UserRole::class,
        'deleted_type' => DeletedType::class,
        'last_access' => 'datetime',
    ];

    protected function password(): Attribute
    {
        return Attribute::make(
            set: fn ($value) => Hash::make($value),
        );
    }

    public function images(): MorphMany
    {
        return $this->morphMany(Image::class, 'imageable');
    }

    public function friendsThatRequestUser()
    {
        return $this->belongsToMany(User::class, 'friends', 'to_user_id', 'from_user_id');
    }

    public function friendsThatUserRequest()
    {
        return $this->belongsToMany(User::class, 'friends', 'from_user_id', 'to_user_id');
    }

    public function unacceptedFriendRequests()
    {
        return $this->friendsThatRequestUser()->wherePivot('is_accepted', false);
    }

    public function getFriends()
    {
        $fromFriends = User::friendsThatRequestUser()->wherePivot('is_accepted', true)->latest('updated_at')->get();
        $toFriends = User::friendsThatUserRequest()->wherePivot('is_accepted', true)->latest('updated_at')->get();

        return $fromFriends->merge($toFriends);
    }

    public function profile(): HasOne
    {
        return $this->hasOne(UserProfile::class);
    }

    public function settings(): BelongsToMany
    {
        return $this->belongsToMany(Setting::class)->withPivot('value')->withTimestamps();
    }

    public function groups(): BelongsToMany
    {
        return $this->belongsToMany(Group::class, 'group_members')
            ->withPivot(['is_seen', 'is_accepted', 'requester'])
            ->withTimestamps()
            ->wherePivot('is_accepted', true);
    }

    public function ownedGroups(): HasMany
    {
        return $this->hasMany(Group::class);
    }

    public function notifications(): HasMany
    {
        return $this->hasMany(Notification::class);
    }

    public function posts(): HasMany
    {
        return $this->hasMany(Post::class)->available(PostAvailableType::LIST);
    }

    public function comments(): HasMany
    {
        return $this->hasMany(Comment::class);
    }

    public function conversations(): BelongsToMany
    {
        return $this->belongsToMany(Conversation::class, 'participants');
    }

    public function postLikes(): BelongsToMany
    {
        return $this->belongsToMany(Post::class, 'post_likes');
    }

    public function commentLikes(): BelongsToMany
    {
        return $this->belongsToMany(Comment::class, 'comment_likes');
    }

    public function scopeAdmin($query)
    {
        $query->where('role', UserRole::ADMIN);
    }

    public function scopeUser($query)
    {
        $query->where('role', UserRole::USER);
    }

    public function isAdmin(): bool
    {
        return $this->role === UserRole::ADMIN;
    }

    public function isUser(): bool
    {
        return $this->role === UserRole::USER;
    }

    public function scopeOwned($query)
    {
        $query->where('id', auth()->id());
    }

    public function scopeFilter($query, $filters)
    {
        $query->when($filters['search'] ?? false, function ($query, $searchText) {
            $query->where('name', 'like', '%' . $searchText . '%')
                ->orWhere('email', 'like', '%' . $searchText . '%')
                ->orWhereHas('profile', function ($query) use ($searchText) {
                    $query->where('first_name', 'like', '%' . $searchText . '%')
                        ->orWhere('last_name', 'like', '%' . $searchText . '%')
                        ->orWhere('country', 'like', '%' . $searchText . '%')
                        ->orWhere('state', 'like', '%' . $searchText . '%')
                        ->orWhere('city', 'like', '%' . $searchText . '%')
                        ->orWhere('address', 'like', '%' . $searchText . '%');
                });
        });
    }
}
