<?php

namespace App\Repositories\Admin;

use Exception;
use Throwable;
use App\Models\Group;
use App\Enums\DeletedType;
use App\Enums\NotificationType;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Repositories\General\NotificationRepository;

class GroupRepository
{
    protected $notificationRepository;

    public function __construct(NotificationRepository $notificationRepository)
    {
        $this->notificationRepository = $notificationRepository;
    }

    public function getAll()
    {
        $groups = Group::with([
            'user',
            'images',
            'category',
        ])
            ->filter(request([
                'search',
            ]))
            ->latest('created_at')
            ->paginate(config('pagination.admin'))
            ->withQueryString();

        return compact('groups');
    }

    public function getTrash()
    {
        $groups = Group::with([
            'user',
            'images',
            'category',
        ])
            ->onlyTrashed()
            ->filter(request([
                'search',
            ]))
            ->latest('created_at')
            ->paginate(config('pagination.admin'))
            ->withQueryString();

        return compact('groups');
    }

    public function show($id)
    {
        $group = Group::with([
            'user',
            'images',
            'category',
        ])
            ->withTrashed()
            ->find($id);

        return compact('group');
    }

    public function destroy(Group $group)
    {
        try {
            DB::beginTransaction();

            $this->notificationRepository->add(
                userId: $group->user_id,
                content: 'Admin decided to remove your group.',
                imageUrl: auth()->user()->images()->profile()->first()?->url,
                type: NotificationType::ADMIN->value,
                redirectLink: '',
            );

            $group->update([
                'deleted_type' => DeletedType::ADMIN,
            ]);
            $group->delete();

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Group deleted failed.');
        }
    }

    public function restore($id)
    {
        try {
            DB::beginTransaction();

            $group = Group::onlyTrashed()->findOrFail($id);

            $group->update([
                'deleted_type' => null,
            ]);
            $group->restore();

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Group restored failed.');
        }
    }
}
