<?php

namespace App\Repositories\General;

use Exception;
use Throwable;
use App\Models\User;
use App\Models\Friend;
use App\Models\Comment;
use App\Enums\DeletedType;
use Illuminate\Http\Request;
use App\Services\ImageService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UserRepository
{
    protected $imageService;

    public function __construct(ImageService $imageService)
    {
        $this->imageService = $imageService;
    }

    public function show($id)
    {
        $user = User::with([
            'profile',
            'images',
        ])->findOrFail($id);

        $friendship = null;
        $loginUserId = auth()->id();
        if ($loginUserId) {
            $friendship = Friend::getByUsers($loginUserId, $user->id)->first();
        }

        return compact('user', 'friendship');
    }

    public function showCount($id)
    {
        $user = User::findOrFail($id);

        $postIds = $user->posts()->pluck('posts.id')->toArray();

        $postsCount = count($postIds);
        $friendsCount = $user->getFriends()->count();

        $postLikesCount = DB::table('post_likes')->whereIn('post_id', $postIds)->count();
        $commentsCount = Comment::whereIn('post_id', $postIds)->count();

        return compact('postsCount', 'commentsCount', 'friendsCount', 'postLikesCount');
    }

    public function showCountDetail($id, $type)
    {
        $user = User::findOrFail($id);

        $loginUser = auth()->user();
        $friends = ($loginUser) ? $loginUser->getFriends()->pluck('id')->toArray() : [];
        $results = collect([]);
        switch ($type) {
            case 'friends':
                $users = $user->getFriends();
                foreach ($users as $user) {
                    $user->is_friend = in_array($user->id, $friends);
                    $results->push($user);
                }
                break;
            case 'post-likes':
                $posts = $user->posts;
                if ($posts->count() > 0) {
                    foreach ($posts as $post) {
                        foreach ($post->likes as $user) {
                            $user->liked_at = $user->pivot->created_at;
                            $user->is_friend = in_array($user->id, $friends);
                            $results->push($user);
                        }
                    }
                }
                $results = $results->sortByDesc('liked_at');
                break;
            case 'comments':
                $posts = $user->posts;
                if ($posts->count() > 0) {
                    foreach ($posts as $post) {
                        foreach ($post->comments as $comment) {
                            $user = $comment->user;
                            $user->commented_at = $comment->updated_at;
                            $user->is_friend = in_array($user->id, $friends);
                            $results->push($user);
                        }
                    }
                }
                $results = $results->sortByDesc('commented_at');
                break;
        }
        $results = $results->paginate(config('pagination.general'))->withQueryString();

        return compact('results');
    }

    public function destroy(Request $request)
    {
        $user = User::owned()->first();

        auth()->logout();

        try {
            DB::beginTransaction();

            $user->update([
                'deleted_type' => DeletedType::USER,
            ]);
            $user->delete();

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('User deleted failed.');
        }

        $request->session()->invalidate();
        $request->session()->regenerateToken();
    }
}
