<?php

namespace App\Repositories\Admin;

use Exception;
use Throwable;
use App\Models\Announcement;
use Illuminate\Http\Request;
use App\Services\ImageService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Repositories\General\NotificationRepository;

class AnnouncementRepository
{
    protected $imageService;
    protected $notificationRepository;

    public function __construct(ImageService $imageService, NotificationRepository $notificationRepository)
    {
        $this->imageService = $imageService;
        $this->notificationRepository = $notificationRepository;
    }

    public function getAll()
    {
        $announcements = Announcement::with([
            'user',
            'image',
        ])
            // TODO: Add or not restore
            // ->withTrashed()
            ->filter(request([
                'search',
            ]))
            ->latest('updated_at')
            ->paginate(12)
            ->withQueryString();

        return compact('announcements');
    }

    public function show($id)
    {
        $announcement = Announcement::with([
            'user',
            'image',
        ])
            ->withTrashed()
            ->find($id);

        return compact('announcement');
    }

    public function save(Request $request)
    {
        $loginUser = auth()->user();
        try {
            DB::beginTransaction();

            $announcement = Announcement::updateOrCreate(
                ['id' => $request->id],
                [
                    'user_id' => $loginUser->id,
                    'title' => $request->title,
                    'description' => $request->description,
                    'link' => $request->link,
                    'start_time' => $request->start_time,
                    'end_time' => $request->end_time,
                ],
            );

            if ($request->hasFile('image')) {
                $oldImage = $announcement->image;
                if ($oldImage) {
                    $this->imageService->delete($oldImage->id, Announcement::class);
                }
                $image = $this->imageService->upload($request->file('image'), Announcement::class);
                $announcement->image()->save($image);
            }

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Announcement saved failed.');
        }
    }

    public function destroy(Announcement $announcement)
    {
        try {
            DB::beginTransaction();

            $announcement->delete();

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Announcement deleted failed.');
        }
    }

    public function restore($id)
    {
        try {
            DB::beginTransaction();

            $announcement = Announcement::onlyTrashed()->findOrFail($id);

            $announcement->restore();

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Announcement restored failed.');
        }
    }
}
